/* $Id: testsca.c,v 1.31 1999/06/21 21:02:34 dhiller Exp $ */
/* Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Don Hiller */

#include "sema.h"

#define ABS(_x_) ((_x_) > 0 ? (_x_) : -(_x_) )

#define CHK(_x_) { if ( (err = (_x_)) < 0 ) goto recover; }
#define ECC(_x_) if ( ! err ) _x_
#define FAIL(_x_) { err = (_x_); goto recover; }

#define NO_SCA_FOUND			-1

#define E1432_24LSBS			0xffffffUL

/* upper half of a24 address space */
#define	E1432_A24_UPPER_1M		0x80000
#define	E1432_A24_UPPER_256K		0x20000
#define	E1432_SCA_SPACING		0x00400	/* bytes */

#define	E1432_FIT_ID_REG		(0x2200L * 4)
#define	E1432_FIT_ID_MASK		0xf8000000UL
#define E1432_CLARION_IDR_BITS		0x38000000UL
#define E1432_CLARINET_IDR_BITS		0x30000000UL
#define	E1432_TACH_ID			0x20000000UL
#define	E1432_TACH_HWREV_MASK		(0x7UL << 24)
#define	E1432_TACH_HWREV		(0x2UL << 24)
#define	E1432_TACH_CONFIG_REG		(0x221cL * 4)

#define	E1432_CLOCK_TRIG_RDIV_REG	(0x210cL * 4)
#define	E1432_MEM_CTL_BUS_CTL_REG	(0x2005L * 4)
#define	E1432_MEM_CTL_BUS_CTL_FORCE_TA	0x2

#define	E1432_SCA_FESR			0x00
#define	E1432_FESR_GND_LO		0x02020202UL
#define	E1432_FESR_CAL_LO		0x03030303UL
#define	E1432_FESR_CAL_HI		0x13131313UL

#define	E1432_SCA_TDR			0xf8
#define	E1432_TDR_BITS			0xffff0000UL
#define	E1432_TDR_WAIT			0x40000000UL /* WAIT bit set, 0 count */

#define	E1432_SCA_CSR			0xfa
#define	E1432_CSR_DIFFS			0x22220000UL

#define	E1432_SCA_ISR			0xfd
#define SCA_ISR_FIXED_MASK		0x00f80000UL /* common to all 1432/3 */
#define	VIBRATO_ISR_FIXED_BITS		0x00080000UL
#define SONATA_ISR_FIXED_BITS		0x00880000UL /* A A1-A3 */
#define SONATA_A_ISR_FIXED_BITS2	0x00c80000UL /* A >= A4 */
#define SCA_ISR_SONATA_B_FIXED_MASK	0x00ffc200UL
#define SONATA_B_ISR_FIXED_BITS_A1	0x00ca0000UL /* B A1 */
#define	E1432_ISR_SUMMARY		0x80000000UL
#define	E1432_ISR_HREQ			0x00040000UL

#define	E1432_SCA_RCR			0xfe
#define E1432_RCR_SOFT_RESET		0x80000000UL


#define	E1432_CAL_CTL			0x8000
#define E1432_CAL_DC_LSBS		0x8030
#define E1432_CAL_DC_MSBS		0x8034

#define E1432_LO_GAIN			(0.316239 * 19.0/15.0)
#define E1432_ADC_FS			1.55


#define	E1433_SCA_HICR			0x14	/* chan A/0, in words */
#define E1433_HSIR_HREQ			0x8100	/* resets, turns on HREQ */

#define	E1433_SCA_HCVR			0x15	/* chan A/0, in words */
#define	E1433_CMD_VECT1			0x8100
#define	E1433_CMD_VECT2			0x8200

#define	E1433_SCA_HISR			0x16	/* chan A/0, in words */
#define	E1433_HISR_BITS			0x0000ff00L
#define	E1433_HISR_DEF			0x00000600L	/* reset state */
#define E1433_HISR_TXDE			0x00000200L

#define	E1433_SCA_HREG_OFFSET		0x10	/* inc to chan B/1, in bytes */
#define	E1433_SCA_HREG_ALL_OFFSET	0x20	/* inc to both chan, in bytes */

#define	E1433_SCA_GCTLR			0x20	/* in words */
#define	E1433_GCTLR_CHAB_DIS		0x00003000L
#define	E1433_GCTLR_DSP_RST		0x00010000L
#define	E1433_GCTLR_CLEAR_READ		0x00020000L
#define	E1433_GCTLR_PACK_MODE		0x00040000L

#define	E1433_SCA_HTX			0x00	/* chan A/0, in words */
#define	E1433_SCA_HTX_OFFSET		0x20	/* inc to chan B/1, in bytes */

#define	E1433_SCA_TDC			0x21	/* in words */
#define	E1433_TDC_BITS			0x0007ffffL

#define	E1433_SCA_GISR			0x22	/* in words */
/* bits that 56k sets + SCA bus error bit */
#define	E1433_SCA_GISR_MASK		0x00000133L

#define E1433_SCA_GXCR			0x23	/* in words */
#define E1433_GXCR_PRE_BITS		0x05	/* PRE_BITS in diag_son.asm */
#define E1433_GXCR_MASK			0x07	/* active bits, chan 0 */
#define E1433_GXCR_POST_BITS		(~E1433_GXCR_PRE_BITS & E1433_GXCR_MASK)
#define E1433_GXCR_CIP			0x00000001L

#define E1433_SCA_ID			0xfd	/* id register */
#define	SUB_ID_A19701025		0x00000200	/* SAMP0 Xilinx */

#define E1433_SCA_DATA			0xff	/* data register */

#define E1433_UP_TEST			0x123456 /* UP_TEST in diag_son.asm */
#define E1433_UP_TEST2			(~E1433_UP_TEST & E1432_24LSBS)


/* margins for overload tests */
#define	E1432_ABOVE_FS_OVLD		1.33
#define	E1432_BELOW_FS_NO_OVLD		 .75

#define	E1432_CAL_CTL_MUX0		0x01
#define	E1432_CAL_CTL_MUX1		0x02
#define	E1432_CAL_CTL_GAIN		0x04
#define	E1432_CAL_CTL_SBIN		0x08
#define	E1432_CAL_CTL_SBOUT		0x10
#define	E1432_CAL_CTL_UPD		0x20
#define	E1432_CAL_CTL_CALIN_MASK	0x0f
#define	E1432_CAL_FULL_SCALE_LO		.4838
#define	E1432_CAL_FULL_SCALE_HI		15.0

static const char epld33_1a[] = "U332";
static const char epld33_1b[] = "U325";
static const char epld33_2a[] = "U362";
static const char epld33_2b[] = "U315";
static const char *const e1432_epld33[4][2] =
  { epld33_2a, epld33_2b, epld33_1a, epld33_1b,
    epld33_1a, epld33_1b, epld33_2a, epld33_2b };


static const unsigned long dsp33acode[] =
{
#include "diag_son_a.asc"
};
static const unsigned long dsp33bcode[] =
{
#include "diag_son_b.asc"
};
static const int dsp33codesize = 512;  /* 56002 starts up after 512 words */

#if 0
static char *
str_bits(unsigned long num, int bits, char *tmp)
{
    int i;

    *tmp = '\0'; /* make tmp "" */

    for ( i = bits-1; i >= 0; i-- )
    {
	if ( num & (1 << bits) ) (void) strcat(tmp, "1");
	else (void) strcat(tmp, "0");
    }

    return tmp;
}
#endif

/* Computes the byte address of the SCA register, assuming page map set
   for the beginning of the SCA address space. */
/* reg is a word address, relative to the base address for that SCA */
static long
sca_addr(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca, int reg)
{
    long    addr;

    if ( mn->a24_256k ) addr = 0; /* only upper window mapped */
    else addr = E1432_A24_UPPER_1M;
    addr += sca * E1432_SCA_SPACING; /* Add in the sca offset */
    addr += reg << 2;	/* Add in register address, X4 for byte address */
    return addr;
}


static SHORTSIZ16
set_cal_voltage(E1432_MODULE_LIST_NODE *mn, double voltage)
{
    SHORTSIZ16 err = 0;
    unsigned long ctl_reg = E1432_CAL_CTL_SBIN;
    double dac_fs;
    long dac;

    if ( ABS(voltage) > E1432_CAL_FULL_SCALE_LO )
    {
	dac_fs = E1432_CAL_FULL_SCALE_HI;
    }
    else
    {
	dac_fs = E1432_CAL_FULL_SCALE_LO;
	ctl_reg |= E1432_CAL_CTL_GAIN;  /* set the low gain bit */
    }
    /* convert to offset binary DAC setting */
    dac = (long)(voltage/dac_fs * 2048.0 + .5) + 2048;

    /* program the dac */
    CHK(i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, E1432_BASE_PAGE));
    i1432_w32(mn, &err, E1432_CAL_CTL, ctl_reg);
    i1432_w32(mn, &err, E1432_CAL_DC_LSBS, dac);
    i1432_w32(mn, &err, E1432_CAL_DC_MSBS, dac >> 8);
    ctl_reg |= E1432_CAL_CTL_UPD;  /* update */
    i1432_w32(mn, &err, E1432_CAL_CTL, ctl_reg);

recover:
    if ( err )
    {
        i1432_print_acc_fail(1, err, "programming substrate CAL circuits");
	DIAG_PRINTF(0, ("  SCA test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    return 0;
}


static SHORTSIZ16
set_sca_page(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca)
{
    SHORTSIZ16 err;
    unsigned short page;

    page = (SHORTSIZ16)
	((sca == E1432_SCA_FIT) ? E1432_FIT_PAGE : E1432_SCA_PAGE);
    if ( mn->a24_256k ) page <<= 2;
    err = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page);
    if ( err )
    {
        i1432_print_acc_fail(1, err, "programming page map register");
	DIAG_PRINTF(0, ("  SCA test failed with main board or VXI problem.\n"));
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    return err;
}


/* variable fields for print_ovld_err() */
static const char gnd_msg[] = "ground selected";
static const char bfs_msg[] = "cal_in below full scale";
static const char afs_msg[] = "cal_in above full scale";
static const char pos_msg[] = "positive ";
static const char neg_msg[] = "negative ";
static const char hi_msg[] = "high";
static const char lo_msg[] = "low";

static void
print_ovld_err(int sca, const char *polarity, const char *selected,
  const char *gain, unsigned long sb, unsigned long is)
{
    DIAG_PRINTF(1, ("  E1432 input SCA %d differential overload failure with\n",
      sca));
    DIAG_PRINTF(1, ("  %s%s, %s gain\n", polarity, selected, gain));
    DIAG_PRINTF(1, ("  overloads should be %4.4lx, were actually %4.4lx\n",
      sb >> 17, (is & E1432_CSR_DIFFS) >> 17));
    DIAG_PRINTF(1, ("                 Ch 4 ^  ^ Ch 1\n"));
}


static SHORTSIZ16
read_24(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 addr, LONGSIZ32 *reg)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 data;
    i1432_r32(mn, &err, addr, &data);
    *reg = (data >> 8) & E1432_24LSBS;
    return err;
}


static SHORTSIZ16
check_id_bus_err(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 err, int expected,
  SHORTSIZ16 sca, const char *sca_msg, SHORTSIZ16 sca_err, SHORTSIZ16 *type)
{
    if ( err == ERR1432_BUS_ERROR )
    {
        *type = E1432_SCA_ID_NONE;
	/* Bus error is OK if not expecting an SCA,
	   reset the module to fix it, then exit */
	i1432_berr_recover(mn);

	/* if ( diag_stop_testing() ) return DIAG_Q_ERR(DIAG_ABORT); */
	/* if ( diagFlags & DIAG_FLAGS_ALLOW_ROM_BOOT ) */ i1432_pause(1.0);
	/* if ( diag_stop_testing() ) return DIAG_Q_ERR(DIAG_ABORT); */
    	if ( expected )
	{
	    DIAG_PRINTF(1, ("  Bus error when trying to read %s ID bits.\n",
	      sca_msg));
	    DIAG_PRINTF(1, ("  No SCA plugged in or Xilinx failure.\n"));
	    DIAG_PRINTF(0, ("  SCA test failed.\n"));
	    return DIAG_Q_ERR(sca_err);
    	}
	DIAG_PRINTF(3, ("    Bus error reading slot %d ID bits (OK).\n",
	  sca + 1));
	return NO_SCA_FOUND;
    }
    else if ( err )
    {
        DIAG_PRINTF(1, ("  Error %d reading SCA id register\n", err));
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
        return ERR1432_DIAG_ERR_SUBST;
    }
    return 0;
}

SHORTSIZ16
i1432_check_sca(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca, SHORTSIZ16 *type)
{
    SHORTSIZ16 err = 0;
    SHORTSIZ16 sca_err;
    unsigned long raw_id_bits = 0;
    unsigned long id_bits = 0;
    unsigned long id_mask;
    SHORTSIZ16 type_found;
    LONGSIZ32 isr_addr = sca_addr(mn, sca, E1432_SCA_ISR);
    LONGSIZ32 idr_addr = sca_addr(mn, sca, 0); /* Clarion ID register */
    /*LONGSIZ32 fit_idr_addr = E1432_FIT_ID_REG;*/
    LONGSIZ32 id_addr;
    const char *sca_msg;
    int expected, input_expected, input_found;
    int fit = (sca == E1432_SCA_FIT);

    /* don't test if supposed to ignore this sca */
    if ( *type ==  E1432_SCA_ID_IGNORE ) return err;

    /* can resolve E1432_SCA_ID_SOURCE to specific source type */
    if ( *type == E1432_SCA_ID_SOURCE )
    {
	if ( fit ) *type = E1432_SCA_ID_CLARINET;
	else *type = E1432_SCA_ID_CLARION;
    }

    /* defaults */
    expected = 1;  /* true unless E1432_SCA_ID_NONE || E1432_SCA_ID_UNKNOWN */
    input_expected = 0; /* only true for E1432_SCA_ID_INPUT */
    id_addr = isr_addr;         /* look for input unless fit/Clarion expected */
    id_mask = SCA_ISR_FIXED_MASK;  /* ditto */

    switch( *type )
    {
    case E1432_SCA_ID_TACH:
	id_addr = E1432_FIT_ID_REG;
	id_mask = E1432_FIT_ID_MASK;
        sca_msg = e1432_tach_str;
        sca_err = ERR1432_DIAG_ERR_TACH;
	break;
    case E1432_SCA_ID_CLARINET:
	id_addr = E1432_FIT_ID_REG;
	id_mask = E1432_FIT_ID_MASK;
        sca_msg = e1432_src_str[sca];
        sca_err = ERR1432_DIAG_ERR_SOURCE_FIT;
	break;
    case E1432_SCA_ID_VIBRATO:
        sca_msg = e1432_in32_str[sca];
	sca_err = e1432_fail32[sca];
	break;
    case E1432_SCA_ID_SONATA_A:
        sca_msg = e1432_in33a_str[sca];
	sca_err = e1432_fail33[sca];
	break;
    case E1432_SCA_ID_SONATA_B:
        sca_msg = e1432_in33b_str[sca];
	sca_err = e1432_fail33[sca];
	break;
    case E1432_SCA_ID_CLARION:
	id_addr = idr_addr;
	id_mask = E1432_FIT_ID_MASK;
        sca_msg = e1432_src_str[sca];
	if ( sca == 0 )
	{
	    sca_err = ERR1432_DIAG_ERR_SOURCE_SCA0;
	}
	else /* sca == 2 */
	{
	    sca_err = ERR1432_DIAG_ERR_SOURCE_SCA1;
	}
	break;
    case E1432_SCA_ID_INPUT:
        input_expected = 1;
        sca_msg = e1432_in_str[sca];
	sca_err = e1432_fail32[sca];
	break;
    case E1432_SCA_ID_NONE:
    case E1432_SCA_ID_UNKNOWN:
    default:
        expected = 0;
	if ( fit )
	{
	    id_addr = E1432_FIT_ID_REG;
	    id_mask = E1432_FIT_ID_MASK;
	    sca_err = ERR1432_DIAG_ERR_SOURCE_FIT;
	}
	else
	{
	    sca_err = e1432_fail32[sca];
	}
        sca_msg = e1432_null_str;
	break;
    }

    if ( fit )
    {
	DIAG_PRINTF(1, ("Checking fit slot.\n"))
    }
    else
    {
	DIAG_PRINTF(1, ("Checking SCA slot %d.\n", sca+1))
    }

    CHK(set_sca_page(mn, sca)); /* set page map to talk to FIT/SCAs */
#ifdef	_HPUX_SOURCE
    /* regular i/o routines do not always reliably handle bus errors on HP-UX */
    err = i1432_direct_ping_register(mn, id_addr);
#endif
    i1432_r32u(mn, &err, id_addr, &raw_id_bits);
    CHK(check_id_bus_err(mn, err, expected, sca, sca_msg, sca_err, type));
    DIAG_PRINTF(3, ("    id_bits @ 0x%lx = 0x%lx\n", id_addr, raw_id_bits))
    id_bits = raw_id_bits & id_mask;

    /* figure out what's actually there (type_found) */
    input_found = 0;
    if ( fit )
    {
        if ( (id_bits & E1432_FIT_ID_MASK) == E1432_TACH_ID )
        {
            type_found = E1432_SCA_ID_TACH;
            DIAG_PRINTF(2, ("  %s found.\n", e1432_tach_str));
        }
        else if ( (id_bits & E1432_FIT_ID_MASK)
          == E1432_CLARINET_IDR_BITS )
        {
            type_found = E1432_SCA_ID_CLARINET;
            DIAG_PRINTF(2, ("  %s found.\n", e1432_src_str[sca]));
        }
        else
        {
            type_found = E1432_SCA_ID_NONE;
            DIAG_PRINTF(2, ("  none present.\n"));
        }
    }
    else
    {
        if ( id_bits == VIBRATO_ISR_FIXED_BITS )
        {
            input_found = 1;
            type_found = E1432_SCA_ID_VIBRATO;
            DIAG_PRINTF(2, ("  %s found.\n", e1432_in32_str[sca]));
        }
        else if ( id_bits == SONATA_ISR_FIXED_BITS )
	{
            input_found = 1;
            type_found = E1432_SCA_ID_SONATA;
            DIAG_PRINTF(1, ("  WARNING, obsolete %s found.\n",
	      e1432_in33a_str[sca]));
	}
        else if ( id_bits == SONATA_A_ISR_FIXED_BITS2 )
        {
	    /* Incudes A4 and beyond Sonata A (all production units) and
	       Sonata B.  Decide whether it's a Sonata B. */
            input_found = 1;
            if ( (raw_id_bits & SCA_ISR_SONATA_B_FIXED_MASK)
	      == SONATA_B_ISR_FIXED_BITS_A1 )
	    {
                type_found = E1432_SCA_ID_SONATA_B;
                DIAG_PRINTF(2, ("  %s found.\n", e1432_in33b_str[sca]));
	    }
	    else
	    {
                type_found = E1432_SCA_ID_SONATA_A;
                DIAG_PRINTF(2, ("  %s found.\n", e1432_in33a_str[sca]));
	    }
        }
        else
        {
            /* check for Clarion */
            id_addr = idr_addr;
            id_mask = E1432_FIT_ID_MASK;
#ifdef  _HPUX_SOURCE
            err = i1432_direct_ping_register(mn, id_addr);
#endif
            i1432_r32u(mn, &err, id_addr, &id_bits);
            CHK(check_id_bus_err(mn, err, expected, sca, sca_msg, sca_err,
              type))
            DIAG_PRINTF(3, ("    id_bits @ 0x%lx = 0x%lx\n", idr_addr, id_bits))
            id_bits &= id_mask;
            if ( id_bits == E1432_CLARION_IDR_BITS )
            {
                type_found = E1432_SCA_ID_CLARION;
                DIAG_PRINTF(2, ("  %s found.\n", e1432_src_str[sca]));
            }
            else
            {
                type_found = E1432_SCA_ID_NONE;
                DIAG_PRINTF(2, ("  none present.\n"));
            }
        }
    }

    if ( type_found != *type )
    {
        if ( ! expected || input_expected && input_found )
	{
	    /* In the *type == E1432_SCA_ID_UNKNOWN and
	       *type == E1432_SCA_ID_NONE cases, we were not necessarily
	       expecting anything, so we take what we get without complaint.
	       In the *type == E1432_SCA_ID_INPUT case, we're happy to take
	       any input types.  */
	    *type = type_found;
	}
	else
	{
	     /* We expected a particular SCA and did not find it */
	    DIAG_PRINTF(1, ("  Unexpected/no board found for %s\n", sca_msg));
	    DIAG_PRINTF(1, ("  or Xilinx/PROM/SCA bus is broken.\n"));
	    DIAG_PRINTF(1, ("  Possibly, model number/options incorrect\n"));
	    *type = E1432_SCA_ID_NONE;
            return DIAG_Q_ERR(sca_err);
	}
    }
    /* else we found what we expected... (*type = type_found) */

recover:
    if ( err == NO_SCA_FOUND )
    {
	/* none expected, none found */
        DIAG_PRINTF(2, ("  none present.\n"));
	err = 0;
    }
    return err;
}


#define OVLD_INDX_MAX 9 /* 1 + 2*2*2 */
/*
 *************************************************************************
 Test (partially) a Vibrato SCA at slot "sca"
 *************************************************************************
 */
SHORTSIZ16
i1432_test_vibrato(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca)
{
    SHORTSIZ16 err = 0;
    SHORTSIZ16 ovld_err;
    int i, j, k;
    const char *sca_msg = e1432_in32_str[sca];
    long fesr_addr = sca_addr(mn, sca, E1432_SCA_FESR);
    long isr_addr = sca_addr(mn, sca, E1432_SCA_ISR);
    /*long tdr_addr = sca_addr(mn, sca, E1432_SCA_TDR);*/
    long rsr_addr = sca_addr(mn, sca, E1432_SCA_RCR);
    long csr_addr = sca_addr(mn, sca, E1432_SCA_CSR);
    unsigned long isr;
    unsigned long ovlds, targ_ovlds;
    double fs, cal_in, polarity;
    const char *pol_msg, *cal_msg, *gain_msg;
    int sca_err = e1432_fail32[sca];
    unsigned long ovld[OVLD_INDX_MAX];
    int ovld_indx = 0;
    char tmp_str[80];

    DIAG_PRINTF(1, ("Testing %s.\n", sca_msg))

    CHK(set_sca_page(mn, sca)); /* set page map to talk to SCA */

#if 0  /* this test is unreliable as it stands... no SAMP0? */
    unsigned long tdr;
    /* test additional data lines by reading TDR */
/*    tdr = lpeek_a24(tdr_addr);	/* to clear */
    i1432_pause(0.01);	/* allow a SAMP0 to go by so the clear to take effect */
/*    tdr = lpeek_a24(tdr_addr);	/* to clear */
    i1432_pause(0.01);	/* allow a SAMP0 to go by so the clear to take effect */
/*    tdr = lpeek_a24(tdr_addr);
    /* should be in the TDR WAIT state - cleared with WAIT set */
    tdr |= E1432_TDR_BITS;
    if ( tdr != E1432_TDR_WAIT )
    {
	DIAG_PRINTF(1, ("  Error reading E1432 input SCA %d TDR.\n", sca));
	DIAG_PRINTF(1, ("  read 0x%8.8lx, should be 0x%8.8lx\n", tdr, E1432_TDR_WAIT));
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
        FAIL(DIAG_Q_ERR(sca_err));
    }
#endif


    /* write testing, 56k testing */

    /* reset DSP, Xilinx, to insure HREQ asserted by DSP */
    i1432_w32(mn, &err, rsr_addr, E1432_RCR_SOFT_RESET);
    /* this is first write, so check for bus error */
    if ( err )
    {
	(void) sprintf(tmp_str, "writing to %s RSR", sca_msg);
        i1432_print_acc_fail(1, err, tmp_str);
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
        FAIL(DIAG_Q_ERR((SHORTSIZ16) sca_err));
    }
    i1432_pause(0.01);
    i1432_w32(mn, &err, rsr_addr, 0);
    i1432_pause(0.01);

    i1432_r32u(mn, &err, isr_addr, &isr);
    if ( ! isr & E1432_ISR_HREQ	)
    {
        DIAG_PRINTF(1, ("  E1432 input SCA %d DSP not asserting HREQ\n", sca));
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
        FAIL(DIAG_Q_ERR((SHORTSIZ16) sca_err));
    }

    /* additionally test writes by effect of isr mask on summary bit */
    /* should not be asserted yet */
    if ( isr & E1432_ISR_SUMMARY )
    {
        DIAG_PRINTF(1, ("  E1432 input SCA %d Xilinx ISR summary bit error\n",
	  sca));
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
        FAIL(DIAG_Q_ERR((SHORTSIZ16) sca_err));
    }
    /* set mask bit so E1432_ISR_HREQ causes IRQ (summary bit set) */
    i1432_w32(mn, &err, isr_addr, E1432_ISR_HREQ);
    i1432_r32u(mn, &err, isr_addr, &isr);
    if ( ! isr & E1432_ISR_SUMMARY )
    {
        DIAG_PRINTF(1, ("  E1432 input SCA %d Xilinx IMR failure\n", sca));
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
        FAIL(DIAG_Q_ERR((SHORTSIZ16) sca_err));
    }

    /* Test partial signal path.
       Drive with substrate DC cal line.
       Detect with overloads.
    */
    /* Unfortunately, the analog differential overload level is too close to
       the ADC overload level to test the ADC.  The ADC can be turned off by
       holding it reset (bit 16 in RCR) but that doesn't help */

    /* first, make sure not overloaded with ground input */
    ovld_err = 0;
    /* ground all channels low gain */
    i1432_w32(mn, &err, fesr_addr, E1432_FESR_GND_LO);
    /* clear overloads by reading */
    i1432_r32u(mn, &err, csr_addr, &ovlds);
    i1432_r32u(mn, &err, csr_addr, &ovlds);
    ovld[ovld_indx++] = ovlds;
    /* should not be overloaded */
    if ( ovlds & E1432_CSR_DIFFS )
    {
	print_ovld_err(sca, "", gnd_msg, lo_msg, 0, ovlds);
        FAIL(DIAG_Q_ERR((SHORTSIZ16) sca_err));
    }

    /* first pass of "i" loop tests with low gain */
    /* cal line, low gain */
    i1432_w32(mn, &err, fesr_addr, E1432_FESR_CAL_LO);
    fs = E1432_ADC_FS/E1432_LO_GAIN;
    gain_msg = lo_msg;
    for ( i = 0; i < 2; i++ )
    {
	/* first pass of "j" loop tests non-overloaded condition */
        cal_in = E1432_BELOW_FS_NO_OVLD * fs;
	targ_ovlds = 0;
	cal_msg = bfs_msg;
	for ( j = 0; j < 2; j++ )
	{
	    /* first pass of "k" loop tests positive overloads */
	    polarity = 1;
	    pol_msg = pos_msg;
            for ( k = 0; k < 2; k++ )
	    {
                CHK(set_cal_voltage(mn, polarity * cal_in));
		/* cal writes change page map */
                CHK(set_sca_page(mn, sca));
                i1432_pause(0.01);
		/* clear overloads by reading */
                i1432_r32u(mn, &err, csr_addr, &ovlds);
                i1432_pause(0.01);
                i1432_r32u(mn, &err, csr_addr, &ovlds);
                if ( ovld_indx < OVLD_INDX_MAX ) ovld[ovld_indx++] = ovlds;
	        if ( (ovlds & E1432_CSR_DIFFS) != targ_ovlds )
	        {
	            print_ovld_err(sca, pol_msg, cal_msg, gain_msg, targ_ovlds,
		      ovlds);
		    /* wait to fail - make more of multiple failures */
                    ovld_err = 1;
	        }
	        polarity = -1;  /* second pass tests negative overloads */
	        pol_msg = neg_msg;
	    }
	    /* second pass of "j" loop tests overloaded condition */
            cal_in = E1432_ABOVE_FS_OVLD * fs;
	    targ_ovlds = E1432_CSR_DIFFS;
	    cal_msg = afs_msg;
	}
	/* second pass of "i" loop tests with high gain */
	/* cal line, 10x gain */
        i1432_w32(mn, &err, fesr_addr, E1432_FESR_CAL_HI);
        fs /= 10.0;  /* high gain is 10x */
        gain_msg = hi_msg;
    }
    /* delayed overload error processing */
    if ( ovld_err )
    {
        unsigned long ovld_and = 0xffffffffUL, ovld_or = 0;
	err = ovld_err;
        for ( i = 0; i < ovld_indx; i++ )
	{
	    ovld_and &= ovld[i];
	    ovld_or |= ovld[i];
	}
	if ( ovld_and & E1432_CSR_DIFFS || ~ovld_or & E1432_CSR_DIFFS )
	{
            DIAG_PRINTF(1, ("  One or more overload bits are stuck.\n"))
            DIAG_PRINTF(1, ("  This may be comparators, the Xilinx,"
	      " or the SCA bus.\n"))
            DIAG_PRINTF(1, ("  If more than one SCA fails with the same"
	      " bits stuck, then the SCA bus\n"))
            DIAG_PRINTF(1, ("  is indicated"
              " and the problem may be on one SCA or the substrate.\n"))
            DIAG_PRINTF(1, ("  Bits (on SCA bus, in hex) stuck high:  "
	      "0x%8.8lx\n", ovld_and & E1432_CSR_DIFFS))
            DIAG_PRINTF(1, ("  Bits (on SCA bus, in hex) stuck low:   "
	      "0x%8.8lx\n", ~ovld_or & E1432_CSR_DIFFS))
            DIAG_PRINTF(1, ("                                      "
	      "Ch 4 ^  ^ Ch 1\n"));
            DIAG_PRINTF(1, ("\n"))
            DIAG_PRINTF(0, ("  SCA test failed.\n"));
            FAIL(DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED));
	}
	else
	{
            DIAG_PRINTF(0, ("  SCA test failed.\n"));
            FAIL(DIAG_Q_ERR((SHORTSIZ16) sca_err));
	}
    }

recover:

    return err;;
}


static SHORTSIZ16
sonata_dsp_test(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca, int chan,
  SHORTSIZ16 type)
{
    SHORTSIZ16 err = 0;
    SHORTSIZ16 dsp_err = 0;
    SHORTSIZ16 sca_err = e1432_fail33[sca];
    LONGSIZ32 tdc_addr = sca_addr(mn, sca, E1433_SCA_TDC);
    LONGSIZ32 hisr_addr =
      sca_addr(mn, sca, E1433_SCA_HISR) + chan * E1433_SCA_HREG_OFFSET;
    LONGSIZ32 htx_addr =
      sca_addr(mn, sca, E1433_SCA_HTX) + chan * E1433_SCA_HTX_OFFSET;
    LONGSIZ32 hcvr_addr =
      sca_addr(mn, sca, E1433_SCA_HCVR) + chan * E1433_SCA_HREG_OFFSET;
    LONGSIZ32 tdc;
    LONGSIZ32 hisr;
    LONGSIZ32 rb1, rb2;
    const LONGSIZ32 *p;
    int sonata_b = (type == E1432_SCA_ID_SONATA_B);
    const char *sca_msg =
      sonata_b ? e1432_in33b_str[sca] : e1432_in33a_str[sca];
    char tmp_str[80];
    int i, j;


    /* read TDC to clear data lines */
    i1432_r32(mn, &err, tdc_addr, &tdc);
    i1432_r32(mn, &err, hisr_addr, &hisr);
    /* check for SCA bus error, too? */
    if ( (hisr & E1433_HISR_BITS) != E1433_HISR_DEF )
    {
        DIAG_PRINTF(1, ("  Error in Host ISR bits on %s, DSP %s \n",
          sca_msg, e1432_dsp33_str[chan]));
        DIAG_PRINTF(1, ("  Read 0x%2.2lx, expected 0x%2.2lx\n",
          (hisr & E1433_HISR_BITS) >> 8, E1433_HISR_DEF >> 8));
        DIAG_PRINTF(1, ("  Problem may be in the Xilinx to DSP interface"
          " or the DSP itself.\n"));
        DIAG_PRINTF(1, ("  This is the first DSP %s register read.\n",
          e1432_dsp33_str[chan]));
        dsp_err = 1;
    }

    if ( ! dsp_err )
    {
        /* download code and test for errors as we go */
	if ( sonata_b )
	{
            p = (LONGSIZ32 *)(&dsp33bcode[0]);
	}
	else
	{
            p = (LONGSIZ32 *)(&dsp33acode[0]);
	}
        for ( i = 0; i < dsp33codesize; i++ )
        {
            j = 100;
            do
            {
                i1432_r32(mn, &err, hisr_addr, &hisr);
            }
            while ( (hisr & E1433_HISR_TXDE) == 0 && j-- > 0 );
            if ( (hisr & E1433_HISR_TXDE) == 0 )
            {
                DIAG_PRINTF(1, ("  Error %s, DSP %s host isr txde not ready,"
		  " word = 0x%x\n",
                  sca_msg, e1432_dsp33_str[chan], i));
                DIAG_PRINTF(1, ("  DSP failed to load program.\n"));
                dsp_err = 1;
                break;
            }
            i1432_w32(mn, &err, htx_addr, *p++ << 8);
        }
    }

    /* test that code is up and running */
    if ( ! dsp_err )
    {
        LONGSIZ32 gxcr_addr = sca_addr(mn, sca, E1433_SCA_GXCR);
        LONGSIZ32 gxcr;
        ULONGSIZ32 gxcr_errs, rb_errs;
        ULONGSIZ32 pre_xpect = E1433_GXCR_PRE_BITS << chan*4;
        ULONGSIZ32 post_xpect = E1433_GXCR_POST_BITS << chan*4;
        LONGSIZ32 pre_bits, post_bits;

        i1432_pause(0.1);
        i1432_r32(mn, &err, gxcr_addr, &gxcr);
        pre_bits = gxcr & (E1433_GXCR_MASK << chan*4);
        (void) read_24(mn, htx_addr, &rb1);
        i1432_pause(0.001);
        (void) read_24(mn, htx_addr, &rb2);
        i1432_pause(0.001);
        i1432_r32(mn, &err, gxcr_addr, &gxcr);
        post_bits = gxcr & (E1433_GXCR_MASK << chan*4);
        gxcr_errs = (pre_bits^pre_xpect) | (post_bits^post_xpect);
        rb_errs = (rb1^E1433_UP_TEST) | (rb2^E1433_UP_TEST2);
        if ( rb_errs || gxcr_errs )
        {
            dsp_err = 1;
            if ( ! rb_errs || ! gxcr_errs )
            {
                DIAG_PRINTF(1, ("  Errors encountered after %s DSP %s"
                  " booted up correctly.\n",
                  sca_msg, e1432_dsp33_str[chan]));
                if ( rb_errs )
                {
                    DIAG_PRINTF(1, ("  Problem is most likely with the"
                      " SCA bus or the Xilinx.\n"));
                }
                else /* gxcr_errs */
                {
                    DIAG_PRINTF(1, ("  Problem is most likely with EPLD"
                      " %s, its address lines or its WR_COND line\n",
                      e1432_epld33[sca][chan]));
                }
            }
            else
            {
                DIAG_PRINTF(1, ("  %s, DSP %s did not boot up"
                  " correctly.\n",
                  sca_msg, e1432_dsp33_str[chan]));
                DIAG_PRINTF(1, ("  Suspect Xilinx to DSP interface"
                  " or DSP itself.\n"));
            }
            if ( rb_errs )
            {
                DIAG_PRINTF(1, ("    test data read back"
                  " incorrectly.\n"));
                DIAG_PRINTF(1, ("      Read 0x%6.6lx, expected 0x%6.6lx,\n",
                  rb1, E1433_UP_TEST, rb1^E1433_UP_TEST));
                DIAG_PRINTF(1, ("      then 0x%6.6lx, expected 0x%6.6lx.\n",
                  rb2, E1433_UP_TEST2));
                DIAG_PRINTF(1, ("      Bits in error = 0x%6.6lx.\n",
                  rb_errs));
                if ( ((rb_errs >> 8) & 0xff) == (rb_errs & 0xff)
                  && ((rb_errs >> 16) & 0xff) == (rb_errs & 0xff) )
                {
                    DIAG_PRINTF(1, ("      Error is the same in each"
                      " DSP hostport byte.\n"));
                }
            }
            if ( gxcr_errs )
            {
                DIAG_PRINTF(1, ("    Xilinx Global Condition Register"
                  " did not get set/read correctly.\n"));
                DIAG_PRINTF(1, ("      Read 0x%2.2lx, expected 0x%2.2lx,\n",
                  pre_bits, pre_xpect));
                DIAG_PRINTF(1, ("      then 0x%2.2lx, expected 0x%2.2lx.\n",
                  post_bits, post_xpect));
                DIAG_PRINTF(1, ("      Bits in error = 0x%2.2lx.\n",
                  gxcr_errs));
            }
        }
    }

    /* check for memory test results
       (Sonata A only - SRAM is entirely internal to 56307) */
    if ( ! dsp_err && ! sonata_b )
    {
        int ram_err = 0, addr_err = 0;
        i1432_pause(.001);
        (void) read_24(mn, htx_addr, &rb1);
        for ( i = 0; rb1 != 0 && i < 10; i++ )
        {
            dsp_err = 1;
            /* test for legitmate failure number */
            if ( (rb1 & 0xfffeec) == 0x100000)
            {
                if ( (rb1 & 0x110) == 0x10 )
                /* data line failure */
                {
                    DIAG_PRINTF(1, ("  SRAM data error %s, DSP %s.\n",
                      sca_msg, e1432_dsp33_str[chan]));
                    (void) read_24(mn, htx_addr, &rb1);
                    DIAG_PRINTF(1, ("     expected: 0x%6.6lx\n", rb1));
                    (void) read_24(mn, htx_addr, &rb1);
                    DIAG_PRINTF(1, ("         read: 0x%6.6lx\n", rb1));
                    ram_err = 1;
                    break;
                }
                else if ( (rb1 & 0x110) == 0x100 )
                /* address line failure */
                {
                    DIAG_PRINTF(1, ("  SRAM address error %s, DSP %s.\n",
                      sca_msg, e1432_dsp33_str[chan]));
                    (void) read_24(mn, htx_addr, &rb1);
                    DIAG_PRINTF(1, ("     expected: 0x%6.6lx\n", rb1));
                    (void) read_24(mn, htx_addr, &rb1);
                    DIAG_PRINTF(1, ("         read: 0x%6.6lx\n", rb1));
                    ram_err = 1;
                    addr_err = 1;
                    break;
                }
            }
            if ( ! ram_err )
            {
                DIAG_PRINTF(1, ("  Error %s, DSP %s did not run"
                  " memory test\n",
                 sca_msg, e1432_dsp33_str[chan]));
                DIAG_PRINTF(1, ("  received 0x%lx\n", rb1));
                break;
            }
            (void) read_24(mn, htx_addr, &rb1);
        }
        if ( ram_err )
        {
            DIAG_PRINTF(1, ("  Key to data lines:\n"));
            DIAG_PRINTF(1, ("               0x-----*"
              "  bits  0 to  3\n"));
            DIAG_PRINTF(1, ("               0x----*-"
              "  bits  4 to  7\n"));
            DIAG_PRINTF(1, ("               0x---*--"
              "  bits  8 to 11\n"));
            DIAG_PRINTF(1, ("               0x--*---"
              "  bits 12 to 15\n"));
            DIAG_PRINTF(1, ("               0x-*----"
              "  bits 16 to 19\n"));
            DIAG_PRINTF(1, ("               0x*-----"
              "  bits 20 to 23\n"));
            if ( addr_err )
            {
                DIAG_PRINTF(1, ("  Note that the bit set in the"
                  " expected data of the address test\n"));
                DIAG_PRINTF(1, ("  is the same as the address line"
                  " toggled.\n"));
            }
        }
    }

    /* next is a test of response to a host vector IRQ */
    if ( ! dsp_err )
    {
	/* ouput host vector 1 */
        i1432_w32(mn, &err, hcvr_addr, E1433_CMD_VECT1);
        i1432_pause(0.01);
	/* outputs E1433_UP_TEST in response */
        (void) read_24(mn, htx_addr, &rb1);
	if ( ! err && rb1 != E1433_UP_TEST )
	{
            dsp_err = 1;
            DIAG_PRINTF(1, ("  Hostport Command Vector error %s, DSP %s.\n",
             sca_msg, e1432_dsp33_str[chan]));
	}
    }

    if ( err )
    {
	(void) sprintf(tmp_str, "testing %s, DSP %s",
		       sca_msg, e1432_dsp33_str[chan]);
        i1432_print_acc_fail(1, err, tmp_str);
	return err;
    }

    if ( dsp_err ) return sca_err;

    return err;
}


static SHORTSIZ16
sonata_data_test(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca)
{
    SHORTSIZ16 err = 0;
    SHORTSIZ16 data_err = 0;
    int i, dsp;
    const char *sca_msg = e1432_in33a_str[sca];
    SHORTSIZ16 sca_err = e1432_fail33[sca];
    LONGSIZ32 rd0, rd1;
    LONGSIZ32 test;
    LONGSIZ32 gctlr_addr = sca_addr(mn, sca, E1433_SCA_GCTLR);
    LONGSIZ32 hicr_addr =
      sca_addr(mn, sca, E1433_SCA_HICR) + E1433_SCA_HREG_ALL_OFFSET;
    LONGSIZ32 hcvr_addr =
      sca_addr(mn, sca, E1433_SCA_HCVR) + E1433_SCA_HREG_ALL_OFFSET;
    LONGSIZ32 data_addr = sca_addr(mn, sca, E1433_SCA_DATA);

    i1432_w32(mn, &err, gctlr_addr, E1433_GCTLR_CLEAR_READ);
    i1432_pause(0.01);
    /* both channels enabled, packed 16 bit mode */
    i1432_w32(mn, &err, gctlr_addr, E1433_GCTLR_PACK_MODE);
    i1432_w32(mn, &err, hicr_addr, E1433_HSIR_HREQ);

    test = 0x1;
    for ( i = 0; i < 32; i++ )
    {
	/* signal output of upper word */
        i1432_w32(mn, &err, hcvr_addr, E1433_CMD_VECT2);
        i1432_pause(0.01);
	/* signal output of lower word */
        i1432_w32(mn, &err, hcvr_addr, E1433_CMD_VECT2);
        i1432_pause(0.01);
        i1432_r32(mn, &err, data_addr, &rd0); /* read chan 0 */
        i1432_r32(mn, &err, data_addr, &rd1); /* read chan 1 */
	if ( rd0 != test || rd1 != test )
	{
	    if ( data_err == 0 )
	    {
                DIAG_PRINTF(1, ("  SCA data bus bit err %s.\n", sca_msg));
	    }
	    if ( data_err == 8 )
	    {
                DIAG_PRINTF(1, ("    ...and more.\n"));
	    }
	    else if (data_err < 8 )
	    {
	        if ( rd0 != test && rd1 != test )
	        {
                    DIAG_PRINTF(1, ("    expected 0x%8.8lx, received"
		      " 0x%8.8lx (DSP0), 0x%8.8lx (DSP1)\n", test, rd0, rd1));
	        }
	        else
	        {
		    dsp = 0;
		    if ( rd1 != test )
		    {
		        /* dsp1, not dsp 0 */
		        dsp = 1;
		        rd0 = rd1;
		    }
                    DIAG_PRINTF(1, ("    expected 0x%8.8lx, received"
	              " 0x%8.8lx (DSP%d)\n", test, rd0, dsp));
	        }
	    }
	    data_err++;
	}
	test <<= 1; /* move "1" up by one bit */
    }

    if ( data_err )
    {
        DIAG_PRINTF(1, ("  Key to SCA data lines:\n"));
        DIAG_PRINTF(1, ("             0x-------*  bits  0 to  3\n"));
        DIAG_PRINTF(1, ("             0x------*-  bits  4 to  7\n"));
        DIAG_PRINTF(1, ("             0x-----*--  bits  8 to 11\n"));
        DIAG_PRINTF(1, ("             0x----*---  bits 12 to 15\n"));
        DIAG_PRINTF(1, ("             0x---*----  bits 16 to 19\n"));
        DIAG_PRINTF(1, ("             0x--*-----  bits 20 to 23\n"));
        DIAG_PRINTF(1, ("             0x-*------  bits 24 to 27\n"));
        DIAG_PRINTF(1, ("             0x*-------  bits 28 to 31\n"));

        return sca_err;
    }

    return err;
}


static SHORTSIZ16
sonata_gisr_test(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca,
  const char *sca_msg, int sca_err)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 gisr_addr = sca_addr(mn, sca, E1433_SCA_GISR);
    LONGSIZ32 gisr;

    /* ID bits were SCA_ISR_FIXED_MASK = 0x00f80000 */
    /* read once to clear, then read cleared value */
    i1432_r32(mn, &err, gisr_addr, &gisr);
    i1432_r32(mn, &err, gisr_addr, &gisr);
    gisr &= E1433_SCA_GISR_MASK;
    if ( ! err && gisr != 0 )
    {
	DIAG_PRINTF(1, ("  Error reading %s Xilinx GISR.\n", sca_msg));
        DIAG_PRINTF(1, ("  Expected 0x00000000, received 0x%8.8lx,\n", gisr));
	DIAG_PRINTF(1, ("  (masked by 0x%8.8lx) on SCA bus.\n",
	  E1433_SCA_GISR_MASK));
	DIAG_PRINTF(1, ("  Suspect Xilinx or SCA bus.\n"));
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
        err = DIAG_Q_ERR((SHORTSIZ16) sca_err);
    }

    return err;
}


static SHORTSIZ16
sonata_tdc_test(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca,
  const char *sca_msg, int sca_err)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 tdc_addr = sca_addr(mn, sca, E1433_SCA_TDC);
    LONGSIZ32 tdc;

    /* read several times first to clear */
    i1432_r32(mn, &err, tdc_addr, &tdc);
    i1432_pause(0.01);	/* allow a SAMP0 to go by so the clear to take effect */
    i1432_r32(mn, &err, tdc_addr, &tdc);
    i1432_pause(0.01);	/* allow a SAMP0 to go by so the clear to take effect */
    i1432_r32(mn, &err, tdc_addr, &tdc);
    /* should be in the TDR WAIT state - cleared */
    if ( ! err && (tdc & E1433_TDC_BITS) != 0 )
    {
	DIAG_PRINTF(1, ("  Error reading %s Xilinx TDC.\n", sca_msg));
	DIAG_PRINTF(1, ("  read 0x%8.8lx, should be 0x%8.8lx\n",
	 tdc & E1433_TDC_BITS, 0));
	DIAG_PRINTF(1, ("  probable Xilinx/SCA bus failure.\n"));
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
        err = DIAG_Q_ERR((SHORTSIZ16) sca_err);
    }

    return err;
}


/* reset DSP to insure that it is in the reset state,
   test for write failure (the first write) */
static SHORTSIZ16
sonata_reset_write_test(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca,
  const char *sca_msg, int sca_err)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 gctlr_addr = sca_addr(mn, sca, E1433_SCA_GCTLR);
    LONGSIZ32 gctlr;
    char tmp_str[80];

    gctlr = E1433_GCTLR_CHAB_DIS;
    gctlr = E1433_GCTLR_CHAB_DIS | E1433_GCTLR_DSP_RST | E1433_GCTLR_CLEAR_READ;
    i1432_w32(mn, &err, gctlr_addr, gctlr);
    /* this is first write, so check for bus error */
    if ( err )
    {
	(void) sprintf(tmp_str, "writing to %s Xilinx", sca_msg);
        i1432_print_acc_fail(1, err, tmp_str);
	DIAG_PRINTF(1, ("  This is the first write to %s.\n", sca_msg));
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
        err = DIAG_Q_ERR((SHORTSIZ16) sca_err);
	return err;
    }
    i1432_pause(0.01);
    /* clear the reset */
    gctlr = E1433_GCTLR_CHAB_DIS;
    i1432_w32(mn, &err, gctlr_addr, gctlr);
    i1432_pause(0.01);

    return err;
}


/*
 *************************************************************************
 Core Sonata test routine (A and B versions).
 *************************************************************************
 */
SHORTSIZ16
test_sonata(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca, SHORTSIZ16 type)
{
    SHORTSIZ16 err = 0;
    SHORTSIZ16 dsp_err;
    int sonata_b = (type == E1432_SCA_ID_SONATA_B);
    const char *sca_msg =
      sonata_b ? e1432_in33b_str[sca] : e1432_in33a_str[sca];
    LONGSIZ32 id_addr = sca_addr(mn, sca, E1433_SCA_ID);
    LONGSIZ32 id;
    int chan;
    int sca_err = e1432_fail33[sca];

    DIAG_PRINTF(1, ("Testing %s.\n", sca_msg))

    /* reset module in order to reset Xilinx */
    /* Xilinx needs to be reset for the TDR (Xilinx sanity) check to work */
    i1432_reset_module(mn);

    CHK(set_sca_page(mn, sca)); /* set page map to talk to SCA */

    /* reset DSP to insure that it is in the reset state,
       test this first write */
    /* (E1433B DSP does not get reset by i1432_reset_module(). */
    CHK(sonata_reset_write_test(mn, sca, sca_msg, sca_err));

    /* test additional SCA bus bits - GISR is bottom 8 */
    CHK(sonata_gisr_test(mn, sca, sca_msg, sca_err));

    /* test additional SCA bus lines by reading TDC */
    CHK(sonata_tdc_test(mn, sca, sca_msg, sca_err));

    /* write testing, 56k testing */

    /* check that the HISR on both DSPS are in reset state */
    for ( chan = 0; chan < 2; chan++ )
    {
	dsp_err = sonata_dsp_test(mn, sca, chan, type);
	if ( dsp_err ) err = dsp_err;
    }
    if ( err )
    {
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
	FAIL(DIAG_Q_ERR(err));
    }

    err = sonata_data_test(mn, sca);
    if ( err )
    {
        DIAG_PRINTF(0, ("  SCA test failed.\n"));
	FAIL(DIAG_Q_ERR(err));
    }

    if ( ! sonata_b )
    {
        /* check, note pre A19701025 PCO boards */
        i1432_r32(mn, &err, id_addr, &id);
        if ( (id & 0x00000200) == 0 )
        {
            DIAG_PRINTF(1, ("  %s is pre A19701025 PCO.\n", sca_msg));
        }
    }

recover:

    return err;
}


/*
 *************************************************************************
 Test (partially) a Sonata A SCA at slot "sca"
 *************************************************************************
 */
SHORTSIZ16
i1432_test_sonata_a(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca)
{
    return test_sonata(mn, sca, E1432_SCA_ID_SONATA_A);
}


/*
 *************************************************************************
 Test (partially) a Sonata B SCA at slot "sca"
 *************************************************************************
 */
SHORTSIZ16
i1432_test_sonata_b(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca)
{
    return test_sonata(mn, sca, E1432_SCA_ID_SONATA_B);
}


/*
 *************************************************************************
 Test the TACH interface.  Slightly.
 *************************************************************************
 */
SHORTSIZ16
i1432_test_tach(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 err = 0;
    unsigned long lVal, bctl;

    DIAG_PRINTF(1, ("Testing TACH interface.\n"))

    /* set page map to talk to FIT */
    (void) set_sca_page(mn, E1432_SCA_FIT);

    i1432_r32u(mn, &err, E1432_FIT_ID_REG, &lVal);

    /* Check rev bits of ID register */
    if ((lVal & E1432_TACH_HWREV_MASK) < E1432_TACH_HWREV)
    {
	DIAG_PRINTF(1, ("  Tach board rev is too small.\n"));
	DIAG_PRINTF(1, ("  expected 0x02000000, got 0x%08lX\n",
	    (lVal & E1432_TACH_HWREV_MASK)));
        DIAG_PRINTF(0, ("  Tach test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_TACH);
    }

    /* reset the tach board before checking status bits */
    i1432_w32(mn, &err, E1432_TACH_CONFIG_REG, 0xc0000000UL);
    if ( err )
    {
        i1432_print_acc_fail(1, err, "writing to Tach CONFIG reg");
        DIAG_PRINTF(1, ("  Tach register reads were OK."
	      "  This was the first attempt to write.\n"));
        DIAG_PRINTF(0, ("  Tach test failed.\n"));
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_TACH);
    }

    i1432_r32u(mn, &err, E1432_FIT_ID_REG, &lVal);
    if ((lVal & 0x00ff0000UL) != 0x00450000UL)
    {
	DIAG_PRINTF(1, ("  Tach board status bits not correct.\n"));
	DIAG_PRINTF(1, ("  expected 0x00450000, got 0x%08lX\n",
	    (lVal & 0x00ff0000UL)));
        DIAG_PRINTF(0, ("  Tach test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_TACH);
    }

    /* Set up Sample3 clock, used by Tach board */
    /* If U384 is messed up, this may cause Sample3 to end up very
       slow, in which case the read of FIT_ID will bus error.  We use
       FORCE TA to ensure that we don't just do a REALLY long bus
       cycle. */
    i1432_w32(mn, &err, E1432_CLOCK_TRIG_RDIV_REG, 0xc00001L);
    if ( err )
    {
	DIAG_PRINTF(1, ("  Diagnostics failed accessing clock/trig RDIVWR"
	  " reg\n"));
        DIAG_PRINTF(0, ("  Tach test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_TACH);
    }

    /* Set up FORCE TA */
    i1432_r32u(mn, &err, E1432_MEM_CTL_BUS_CTL_REG, &bctl);
    if ( ! err )
    {
        bctl |= E1432_MEM_CTL_BUS_CTL_FORCE_TA;
        i1432_w32(mn, &err, E1432_MEM_CTL_BUS_CTL_REG, bctl);
    }
    if ( err )
    {
	DIAG_PRINTF(1, ("  Diagnostics failed accessing bus ctl reg\n"));
	DIAG_PRINTF(1, ("\n"));
	DIAG_PRINTF(1, ("  Suspect Xilinx U708, or rom U705\n"));
        DIAG_PRINTF(0, ("  Tach test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_TACH);
    }

    /* Read FIT ID again */
    i1432_r32u(mn, &err, E1432_FIT_ID_REG, &lVal);
    if ( err )
    {
	DIAG_PRINTF(1, ("  Diagnostics failed.  Got bus error accessing"
	  " Tach ID\n"));
	DIAG_PRINTF(1, ("  register when FORCE TA was set, after setting"
	  " up Sample3.\n"));
	DIAG_PRINTF(1, ("  Tach ID was OK and no bus error when\n"));
	DIAG_PRINTF(1, ("  FORCE TA and Sample3 were not set.\n"));
	DIAG_PRINTF(1, ("\n"));
	DIAG_PRINTF(1, ("  Suspect Xilinx U708, or rom U705.\n"));
	DIAG_PRINTF(1, ("  Less likely, suspect slow Sample3, due to bad"
	  " U384.\n"));
	DIAG_PRINTF(1, ("  Sample3 should be 4.9142 MHz.\n"));
        DIAG_PRINTF(0, ("  Tach test failed.\n"));
	err = DIAG_Q_ERR(ERR1432_DIAG_ERR_TACH);
	goto cleanup;
    }
    if ((lVal & E1432_FIT_ID_MASK) != E1432_TACH_ID)
    {
	DIAG_PRINTF(1, ("  Diagnostics failed.  Got wrong ID accessing Tach"
	  " ID\n"));
	DIAG_PRINTF(1, ("  register when FORCE TA was set, after setting"
	  " up\n"));
	DIAG_PRINTF(1, ("  Sample3.  Tach ID was OK when FORCE TA and"
	  " Sample3\n"));
	DIAG_PRINTF(1, ("  were not set.\n"));
	DIAG_PRINTF(1, ("\n"));
	DIAG_PRINTF(1, ("  Suspect slow Sample3, due to bad U384.\n"));
	DIAG_PRINTF(1, ("  Sample3 should be 4.9142 MHz.\n"));
        DIAG_PRINTF(0, ("  Tach test failed.\n"));
	err = DIAG_Q_ERR(ERR1432_DIAG_ERR_TACH);
	goto cleanup;
    }

    /* Clean up force TA */
 cleanup:
    bctl &= ~E1432_MEM_CTL_BUS_CTL_FORCE_TA;
    i1432_w32(mn, &err, E1432_MEM_CTL_BUS_CTL_REG, bctl);
    if ( err )
    {
	DIAG_PRINTF(1, ("  Diagnostics failed accessing bus ctl reg\n"));
	DIAG_PRINTF(1, ("\n"));
	DIAG_PRINTF(1, ("  Suspect Xilinx U708, or rom U705\n"));
        DIAG_PRINTF(0, ("  Tach test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_TACH);
    }
    return err;
}
